function [Ps, s_states, tcounts] = training_opt(update_plan, call_length, dates, s)
% TRAINING_OPT  Train the reality-mining dataset
%   [Ps, s_states, tcounts] = training_opt(update_plan, call_length, dates, s) 
%     return: the P matrix, unique list of transitions, and transition counts
%     
%     update_plan: 1 = always-update,  2 = LA only, 3 = Forming LA
%     call_length: assumed call length
%     dates: date ranges trained
%     s: reality-mining data structure

ad_total = 1013; % number of unique locations, used for smoothing
Ps = {}; % list of P matrices
tcounts = {};
s_states = {}; % list of cellid associations with P matrix col/row

for n = 1:size(s,2) % iterate through every user
    if size(s(n).locs,1) == 0 % skip users with no location data
        continue;
    end;
    
    call_times = [s(n).comm_voice_date, s(n).comm_voice_date + call_length];
    
    % consider only transitions within date range
    locids = s(n).locs(dates(1) < s(n).locs(:,1) & s(n).locs(:,1) < dates(2),:);
    
    % ignore null locids (zero)
    locids(locids(:,2)==0,:) = []; 
    
    % need at least one locid
    if size(locids,1) < 2
        continue;
    end;
    
    if update_plan ~= 1
        if isempty(call_times) || update_plan == 2
            locids = floor(locids); % LACs only
        else
            for t = 1:size(locids,1)
                if ~sum(call_times(:,1) < locids(t,1) & locids(t,1) < call_times(:,2))
                    % next conditional if fullids after lac change
                    if (t > 1 && floor(locids(t,2)) == floor(locids(t-1,2)))
                        locids(t,2) = floor(locids(t,2));
                    end;
                    
                end;
            end;
        end;
    end;
    
    % convert to two columns for transitions
    transitions = [locids(1:end-1,2) locids(2:end,2)];

    % filter out same transitions
    transitions(transitions(:,1) == transitions(:,2),:) = [];

    % at least one unique transition
    if size(transitions,1) < 1
        continue;
    end;
    
    states = unique(locids(:,2)); % unique list of transitions we will consider
    
    % get unique rows
    [u, ~, ic] = unique(transitions, 'rows');        

    % get counts of unique rows
    trans_counts = [u, histc(ic, 1:length(u))];    
    
    % initialize count matrix
    transition_counts = zeros(size(states,1),size(states,1)); 
    
    for pr = 1:size(trans_counts,1) % count transitions
        st_i = trans_counts(pr,1);
        st_j = trans_counts(pr,2);
        
        transition_counts(states==st_i,states==st_j) = trans_counts(pr,3);
    end;
    
    % store states to list for all users
    s_states{n} = states;
    
    % divide each row by the sum of the row and store to matrix list
    tcounts{n} = transition_counts;
    Ps{n} = diag(1./(sum(transition_counts,2) + ad_total))*transition_counts;
    
end;
